import { BrowserModule } from '@angular/platform-browser';
import { APP_INITIALIZER, NgModule } from '@angular/core';
import { BrowserAnimationsModule } from '@angular/platform-browser/animations';

import { AppRoutingModule } from './app.routing.module';
import { AppComponent } from './app.component';
import { ProcessListComponent } from "./components/process-list/process-list.component";
import { HttpClientModule, HttpClient } from '@angular/common/http';
import { TranslateLoader, TranslateModule } from '@ngx-translate/core';
import { TranslateHttpLoader } from '@ngx-translate/http-loader';
import { MaterialModule } from './material.module';
import { FormsModule } from './forms/forms.module';
import { UserComponent } from "./components/user/user.component";
import { FontAwesomeModule } from '@fortawesome/angular-fontawesome';
import {AppCoreModule, FormService as CoreFormService, SecurityService as CoreSecurityService} from "@pas/app-core";
import { FormElementsModule } from "@pas/form-elements";
import {XumlService} from "@pas/xuml-communication";
import {
    HelperService,
    PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL,
    KeyCloakHelperService
} from "@pas/platform-communication";
import {FormService} from "./services/form.service";
import {SecurityService} from "./services/security.service";
import {LibrariesModule} from "./libraries.module";
import {AppConfig} from "./app-config";

// AoT requires an exported function for factories
export function HttpLoaderFactory(http: HttpClient) {
    return new TranslateHttpLoader(http, './assets/i18n/', '.json');
}

@NgModule({
    declarations: [
        AppComponent,
        ProcessListComponent,
        UserComponent
    ],
    imports: [
        AppCoreModule,
        MaterialModule,
        FontAwesomeModule,
        FormElementsModule,
        BrowserModule,
        BrowserAnimationsModule,
        AppRoutingModule,
        HttpClientModule,
        TranslateModule.forRoot({
            loader: {
                provide: TranslateLoader,
                useFactory: HttpLoaderFactory,
                deps: [HttpClient]
            },
            defaultLanguage: 'en'
        }),
        LibrariesModule,
        FormsModule
    ],
    providers: [
        AppComponent,
        {provide: CoreFormService, useClass: FormService},
        {provide: CoreSecurityService, useClass: SecurityService},
        {
            provide: PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL,
            useFactory: (config: AppConfig) => {
                return (config.platformBaseUrl != null) ? config.platformBaseUrl : HelperService.getBaseUrl();
            },
            deps: [AppConfig]
        },
        {
            provide: XumlService,
            useFactory: (config: AppConfig, platformBaseUrl) => {
                let serviceBaseUrl = undefined;
                if(config.xuml.url){
                    serviceBaseUrl = config.xuml.url;
                } else if(config.xuml.platformRelativeUrl){
                    serviceBaseUrl = platformBaseUrl + config.xuml.platformRelativeUrl
                }
                return new XumlService(serviceBaseUrl);
            },
            deps: [AppConfig, PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL]

        },
        {
            provide: APP_INITIALIZER,
            useFactory: initializeKeycloak,
            multi: true,
            deps: [AppConfig, KeyCloakHelperService, XumlService]
        }
    ],
    bootstrap: [AppComponent]
})
export class AppModule { }

function initializeKeycloak(config: AppConfig, keycloakHelper: KeyCloakHelperService, xumlService: XumlService) {
    return async () => {
        try {
            await keycloakHelper.initKeycloak(config.keyCloakOptions);
            if(!config.xuml.isAuthenticator) {
                xumlService.addAxiosInterceptor(keycloakHelper.getAxiosInterceptor());
            }
        } catch (error) {
            console.error('Error initializing Keycloak', error);
        }
    };
}
