/*
 * e2e technologies Ltd.
 * Project:
 * Copyright &#169; 2001-2005 e2e technologies Ltd. All rights reserved.
 * $Id: ZipTools.java,v 1.3 2010/11/19 11:06:53 pmuessig Exp $
 */
package ch.e2e.bridge.tools.zip;

import ch.e2e.bridge.server.BridgeException;

import java.io.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Some usefull method to handle packed content within the E2E Bridge.<br><br>
 * <B>This class can be used to import.</B><br><br>
 * You can use the methods:<br>
 * <ul>
 * <code>
 * <li><a href="#getZipEntries(java.lang.String)">public String[] getZipEntries(String zipFilename)</a></li>
 * <li><a href="#readAllFromZip(java.lang.String)">public ZipObject[] readAllFromZip(String zipFilename)</a></li>
 * <li><a href="#readFromZip(java.lang.String)">public ZipObject readFromZip(String zipFilename)</a></li>
 * <li><a href="#readFromZip(java.lang.String, java.lang.String)">ZipObject readFromZip(String zipFilename, String zipEntryName)</a></li>
 * <li><a href="#writeToZip(java.lang.String, java.io.InputStream)">public void writeToZip(String zipFilename, InputStream from)</a></li>
 * <li><a href="#writeToZip(java.lang.String, java.lang.String, java.io.InputStream)">public void writeToZip(String zipFilename, String entryName, InputStream from)</a></li>
 * <li><a href="#writeToZip(java.lang.String, ch.e2e.bridge.tools.zip.ZipObject)">public void writeToZip(String zipFilename, ZipObject[] zipObjects)</a></li>
 * <li><a href="#unzip(java.io.InputStream)">public ZipObject[] unzip(InputStream in)</a></li>
 * <li><a href="#zip(ch.e2e.bridge.tools.zip.ZipObject[])">public byte[] zip(ZipObject[] in)</a></li>
 * <li><a href="#zip(ch.e2e.bridge.tools.zip.ZipObject[], java.lang.String)">public byte[] zip(ZipObject[] in, String comment)</a></li>
 * <li><a href="#zip(ch.e2e.bridge.tools.zip.ZipObject[], int)">public byte[] zip(ZipObject[] in, int level)</a></li>
 * <li><a href="#zip(ch.e2e.bridge.tools.zip.ZipObject[]), java.lang.String, int">public byte[] zip(ZipObject[] in, String comment, int level)</a></li>
 * </code>
 * </ul>
 * Copyright &#169; 2001-2005 E2E technologies Ltd. All rights reserved.
 * @author tzimber
 * @version $Revision: 1.3 $
 */
@SuppressWarnings({"UnusedDeclaration"})
public class ZipTools {
	private final static int BUFF_SIZE = 100000;

	/**
	 * Extracts the first entry from the given zipfile.
	 * @param zipFilename The packed file.
	 */
	public static ZipObject readFromZip(String zipFilename) {
		ZipObject result = new ZipObject();
		ZipFile zipfile = null;
		try {
			zipfile = new ZipFile(zipFilename);
			Enumeration<? extends ZipEntry> entries = zipfile.entries();
			if (entries.hasMoreElements()) {
				ZipEntry zipEntry = entries.nextElement();
				if (!zipEntry.isDirectory()) {
					ByteArrayOutputStream out = new ByteArrayOutputStream();
					copy(zipfile.getInputStream(zipEntry), out, true, true);
					result.setBlob(out.toByteArray());
				}
				result.setDirectory(zipEntry.isDirectory());
				result.setName(zipEntry.getName());
			} else {
				throw new BridgeException("No entry found in packed file '"
						+ zipFilename + "'.");
			}
		} catch (IOException e) {
			throw new BridgeException("Error reading from packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception e) {
				// ignore
			}
		}
		return result;
	}

	/**
	 * Extracts the entry with the given name from the given zipfile.
 	 * @param zipFilename The packed file.
	 * @param zipEntryName The name of the entry to extract.
	 */
	public static ZipObject readFromZip(String zipFilename, String zipEntryName) {
		ZipObject result = new ZipObject();
		ZipFile zipfile = null;
		try {
			zipfile = new ZipFile(zipFilename);
			ZipEntry zipEntry = zipfile.getEntry(zipEntryName);
			if (zipEntry == null) {
				throw new BridgeException("Entry '" + zipEntryName
						+ "' not found in packed file '"
						+ zipFilename + "'.");
			} else {
				if (!zipEntry.isDirectory()) {
					ByteArrayOutputStream out = new ByteArrayOutputStream();
					copy(zipfile.getInputStream(zipEntry), out, true, true);
					result.setBlob(out.toByteArray());
				}
				result.setDirectory(zipEntry.isDirectory());
				result.setName(zipEntryName);
			}
		} catch (IOException e) {
			throw new BridgeException("Error reading from packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception e) {
				// ignore
			}
		}
		return result;
	}

	/**
	 * Uncompresses the incoming data stream.
	 * @param in The compressed data stream
	 * @return the uncompressed data stream
	 */
	public static ZipObject[] unzip(InputStream in) {
		try {
			return readAllFromZip(in);
		} catch (IOException e) {
			throw new BridgeException("Error reading from compressed Blob.", e);
		} finally {
			try {
				if (in != null) {
					in.close();
				}
			} catch (Exception e) {
				// ignore
			}
		}
	}

	/**
	 * Compresses the incoming data.
	 * @param zipObjects The objects containing data and the names of the zip entries.
	 * @return the compressed data stream
	 */
	public static byte[] zip(ZipObject[] zipObjects) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ZipOutputStream zipout = new ZipOutputStream(out);
		copyZipObjects(zipObjects, zipout);
		try {
			zipout.close();
		} catch (IOException e) {
			throw new BridgeException("Error writing compressed ZipObjects.", e);
		}
		return out.toByteArray();
	}

	/**
	 * Compresses the incoming data.
	 * @param zipObjects The objects containing data and the names of the zip entries.
	 * @param comment The documentation of the compressed stream.
	 * @return the compressed data stream
	 */
	public static byte[] zip(ZipObject[] zipObjects, String comment) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ZipOutputStream zipout = new ZipOutputStream(out);
		zipout.setComment(comment);
		copyZipObjects(zipObjects, zipout);
		try {
			zipout.close();
		} catch (IOException e) {
			throw new BridgeException("Error writing compressed ZipObjects.", e);
		}
		return out.toByteArray();
	}

	/**
	 * Compresses the incoming data.
	 * @param zipObjects The objects containing data and the names of the zip entries.
	 * @param level The compression level (0-9)
	 * @return the compressed data stream
	 */
	public static byte[] zip(ZipObject[] zipObjects, int level) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ZipOutputStream zipout = new ZipOutputStream(out);
		zipout.setLevel(level);
		copyZipObjects(zipObjects, zipout);
		try {
			zipout.close();
		} catch (IOException e) {
			throw new BridgeException("Error writing compressed ZipObjects.", e);
		}
		return out.toByteArray();
	}

	/**
	 * Compresses the incoming data.
	 * @param zipObjects The objects containing data and the names of the zip entries.
	 * @param comment The documentation of the compressed stream.
	 * @param level The compression level (0-9)
	 * @return the compressed data stream
	 */
	public static byte[] zip(ZipObject[] zipObjects, String comment, int level) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ZipOutputStream zipout = new ZipOutputStream(out);
		zipout.setComment(comment);
		zipout.setLevel(level);
		copyZipObjects(zipObjects, zipout);
		try {
			zipout.close();
		} catch (IOException e) {
			throw new BridgeException("Error writing compressed ZipObjects.", e);
		}
		return out.toByteArray();
	}

	private static void copyZipObjects(ZipObject[] zipObjects, ZipOutputStream zipout) {
		for (ZipObject zipObject : zipObjects) {
			copyToZip(zipObject, zipout);
		}
	}

	private static void copyToZip(ZipObject zipObject, ZipOutputStream zipOut) {
		try {
			zipOut.putNextEntry(new ZipEntry(zipObject.getName()));
			if (!zipObject.isDirectory()) {
				copy(new ByteArrayInputStream(zipObject.getBlob()), zipOut, true, false);
			}
			zipOut.closeEntry();
		} catch (IOException e) {
			throw new BridgeException("Error writing compressed Blob.", e);
		}
	}

	/**
	 * Extracts the entry with the given name from the given zipfile.
	  * @param zipFilename The packed file.
	 */
	public static ZipObject[] readAllFromZip(String zipFilename) {
		try {
			return readAllFromZip(new FileInputStream(zipFilename));
		} catch (IOException e) {
			throw new BridgeException("Error reading from packed file: '"
					+ zipFilename + "'.", e);
		}

	}


	private static ZipObject[] readAllFromZip(InputStream in) throws IOException {
		ArrayList<ZipObject> objects = new ArrayList<ZipObject>();
		ZipInputStream zipIn = new ZipInputStream(in);
		ZipEntry entry = zipIn.getNextEntry();
		while (entry != null) {
			ZipObject object = new ZipObject(entry.getName());
			object.setDirectory(entry.isDirectory());
			if (!entry.isDirectory()) {
				ByteArrayOutputStream out = new ByteArrayOutputStream();
				copy(zipIn, out, false, true);
				object.setBlob(out.toByteArray());
			}
			zipIn.closeEntry();
			objects.add(object);
			entry = zipIn.getNextEntry();
		}
		ZipObject[] result = new ZipObject[objects.size()];
		objects.toArray(result);
		return result;
	}

	/**
	 * Packs the given objects in the output file.
	 * @param zipFilename The file to write the packed data.
	 * @param zipObjects The objects to pack.
	 */
	public static void writeToZip(String zipFilename, ZipObject[] zipObjects) {
		ZipOutputStream zipfile = null;
		try {
			zipfile = new ZipOutputStream(new FileOutputStream(zipFilename));
			for (ZipObject zipObject : zipObjects) {
				ZipEntry entry = new ZipEntry(zipObject.getName());
				zipfile.putNextEntry(entry);
				copy(new ByteArrayInputStream(zipObject.getBlob()), zipfile, true, false);
				zipfile.closeEntry();
			}
		} catch (Exception e) {
			throw new BridgeException("Error writing to packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception ex) {
				// ignore
			}
		}
	}

	/**
	 * Packs the given input to an entry with the given name in the output file
	 * @param zipFilename The file to write the packed data.
	 * @param entryName The name of the entry to use in the packed file.
	 * @param from The stream to read the data from.
	 */
	public static void writeToZip(String zipFilename, String entryName, InputStream from) {
		ZipOutputStream zipfile = null;
		try {
			zipfile = new ZipOutputStream(new FileOutputStream(zipFilename));
			ZipEntry entry = new ZipEntry(entryName);
			zipfile.putNextEntry(entry);
			copy(from, zipfile, true, false);
			zipfile.closeEntry();
		} catch (Exception e) {
			throw new BridgeException("Error writing to packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception ex) {
				// ignore
			}
			try {
				from.close();
			} catch (Exception ex) {
				// ignore
			}
		}
	}

	/**
	 * Packs the given input to the output file
	 * @param zipFilename The file to write the packed data.
	 * @param from The stream to read the data from.
	 */
	public static void writeToZip(String zipFilename, InputStream from) {
		ZipOutputStream zipfile = null;
		try {
			zipfile = new ZipOutputStream(new FileOutputStream(zipFilename));
			ZipEntry entry = new ZipEntry("packed");
			zipfile.putNextEntry(entry);
			copy(from, zipfile, true, false);
			zipfile.closeEntry();
		} catch (Exception e) {
			throw new BridgeException("Error writing to packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception ex) {
				// ignore
			}
			try {
				from.close();
			} catch (Exception ex) {
				// ignore
			}
		}
	}

	/**
	 * Retrieves a list of all entries in the packed file.
	 * @param zipFilename The name of the zipfile to search for.
	 * @return a list with all entries of the packed file.
	 */
	public static String[] getZipEntries(String zipFilename) {
		ArrayList<String> entries = new ArrayList<String>();
		ZipFile zipfile = null;
		try {
			zipfile = new ZipFile(zipFilename);
			Enumeration<? extends ZipEntry> fileEntries = zipfile.entries();
			while (fileEntries.hasMoreElements()) {
				ZipEntry entry = fileEntries.nextElement();
				entries.add(entry.getName());
			}
		} catch (IOException e) {
			throw new BridgeException("Error reading from packed file: '"
					+ zipFilename + "'.", e);
		} finally {
			try {
				if (zipfile != null) {
					zipfile.close();
				}
			} catch (Exception e) {
				// ignore
			}
		}
		String[] result = new String[entries.size()];
		entries.toArray(result);
		return result;
	}

	/**
	 * Copies the stream &quot;from&quot; to the stream &quot;to&quot;.
	 * This method can be used for every stream.
	 * @param from The stream to copy from.
	 * @param to The stream to copy to.
	 * @throws IOException If an error occurs.
	 */
	private static void copy(InputStream from, OutputStream to, boolean closeFrom, boolean closeTo)
			throws IOException {
		byte[] buffer = new byte[BUFF_SIZE];
		try {
			while (true) {
				synchronized (buffer) {
					int amountRead = from.read(buffer);
					if(amountRead == -1) {
						break;
					}
					to.write(buffer, 0, amountRead);
				}
			}
		} finally {
			if (closeFrom) {
				try {
					from.close();
				} catch(Exception ex) {
					// ignore
				}
			}
			if (closeTo) {
				try {
					to.close();
				} catch(Exception ex) {
					// ignore
				}
			}
		}
	}
}
