/*
 * e2e technologies Ltd.
 * Project:
 * Copyright &#169; 2001-2005 e2e technologies Ltd. All rights reserved.
 * $Id: DateTools.java,v 1.5 2010/12/20 09:11:03 pmuessig Exp $
 */
package ch.e2e.examples.date;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * This class offers you a few methods to show you how to handle with
 * java.util.Date and java.util.Calendar objects in the E2E Bridge.<br><br>
 * <B>This class can be used to import.</B><br><br>
 * You can use the method:<br>
 * <ul>
 * <code><li><a href="#getDate()">public Date getDate()</a></li></code>
 * <code><li><a href="#getCalendar()">public GregorianCalendar getCalendar()</a></li></code>
 * <code><li><a href="#getTestObject()">public DateContainer getTestObject()</a></li></code>
 * <code><li><a href="#getSundays(int)">public GregorianCalendar[] getSundays(int year)</a></li></code>
 * <code><li><a href="#getSaturdays(int)">public List getSaturdays(int year)</a></li></code>
 * <code><li><a href="#getWeekends(int)">public Map getWeekends(int year)</a></li></code>
 * <code><li><a href="#getDayOfWeek(GregorianCalendar)">public String getDayOfWeek(GregorianCalendar date)</a></li></code>
 * <code><li><a href="#sortByDayOfWeek(ArrayList)">public Date[] sortByDayOfWeek(ArrayList dates)</a></li></code>
 * <code><li><a href="#sortByDayOfYear(Date[])">public Date[] sortByDayOfYear(Date[] dates)</a></li></code>
 * </ul>
 * Copyright &#169; 2001-2005 e2e technologies Ltd. All rights reserved.
 * @author tzimber
 * @version $Revision: 1.5 $
 */
@SuppressWarnings({"UnusedDeclaration"})
public class DateTools {
	/**
	 * Returns today as a java.util.Date object.
	 * @return today as a java.util.Date object.
	 */
	public static Date getDate() {
		return new Date();
	}

	/**
	 * Returns today as a java.util.GregorianCalendar object.
	 * @return today as a java.util.GregorianCalendar object.
	 */
	public static GregorianCalendar getCalendar() {
		return new GregorianCalendar();
	}

	/**
	 * Returns an object with a set of different Date and Calendar attributes.
	 * @return an object with a set of different Date and Calendar attributes.
	 */
	public static DateContainer getDateContainer() {
		DateContainer result = new DateContainer();
		GregorianCalendar today = new GregorianCalendar();
		result.setCalendar(today);
		GregorianCalendar tomorrow = new GregorianCalendar(today.get(Calendar.YEAR),
				today.get(Calendar.MONTH), today.get(Calendar.DAY_OF_MONTH) + 1);
		GregorianCalendar[] calendars = new GregorianCalendar[]{today, tomorrow};
		result.setCalendars(calendars);
		result.setCalendarList(Arrays.asList(calendars));
		HashMap<String, GregorianCalendar> calendarMap = new HashMap<String, GregorianCalendar>();
		calendarMap.put("today", today);
		calendarMap.put("tomorrow", tomorrow);
		result.setCalendarMap(calendarMap);
		TreeSet<GregorianCalendar> calendarSet = new TreeSet<GregorianCalendar>(new Comparator<GregorianCalendar>() {
			public int compare(GregorianCalendar o1, GregorianCalendar o2) {
				return (o1.getTime().compareTo(o2.getTime()));
			}
		});
		calendarSet.addAll(Arrays.asList(calendars));
		result.setCalendarSet(calendarSet);
		result.setDate(today.getTime());
		Date lastMonth = new GregorianCalendar(today.get(Calendar.YEAR),
				today.get(Calendar.MONTH) - 1, today.get(Calendar.DAY_OF_MONTH)).getTime();
		Date lastYear = new GregorianCalendar(today.get(Calendar.YEAR) - 1,
				today.get(Calendar.MONTH), today.get(Calendar.DAY_OF_MONTH)).getTime();
		Date[] dates = new Date[]{lastMonth, lastYear};
		result.setDates(dates);
		result.setDateList(Arrays.asList(dates));
		HashMap<String, Date> dateMap = new HashMap<String, Date>();
		dateMap.put("lastMonth", lastMonth);
		dateMap.put("lastYear", lastYear);
		result.setDateMap(dateMap);
		result.setDateSet(new TreeSet<Date>(Arrays.asList(dates)));
		return result;
	}

	/**
	 * Returns an array of all sundays of the given year.
	 * @param year The year to retrieve the sundays.
	 * @return an array of all sundays of the given year.
	 */
	public static GregorianCalendar[] getSundays(int year) {
		ArrayList<GregorianCalendar> sundayList = new ArrayList<GregorianCalendar>();
		GregorianCalendar day = new GregorianCalendar(year, 0, 1);
		int daysInYear = day.isLeapYear(year)? 366: 365;
		for (int i = 0; i < daysInYear; i++) {
			day.set(Calendar.DAY_OF_YEAR, (i + 1));
			if (day.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY) {
				GregorianCalendar sunday = new GregorianCalendar();
				sunday.setTime(day.getTime());
				sundayList.add(sunday);
			}
		}
		GregorianCalendar[] result = new GregorianCalendar[sundayList.size()];
		sundayList.toArray(result);
		return result;
	}

	/**
	 * Returns a list of all saturdays of the given year.
	 * @param year The year to retrieve the saturdays.
	 * @return a list of all saturdays of the given year.
	 */
	public static List<Date> getSaturdays(int year) {
		ArrayList<Date> saturdayList = new ArrayList<Date>();
		GregorianCalendar day = new GregorianCalendar(year, 0, 1);
		int daysInYear = day.isLeapYear(year)? 366: 365;
		for (int i = 0; i < daysInYear; i++) {
			day.set(Calendar.DAY_OF_YEAR, (i + 1));
			if (day.get(Calendar.DAY_OF_WEEK) == Calendar.SATURDAY) {
				saturdayList.add(day.getTime());
			}
		}
		return saturdayList;
	}

	/**
	 * Returns a map of all weekends in the given year. The key is of type
	 * <code>java.util.Date</code> and the value is of type
	 * <code>java.lang.String</code>.
	 * @param year The year to retrieve the weekends.
	 * @return  a map of all weekends in the given year.
	 */
	public static Map<Date, String> getWeekends(int year) {
		TreeMap<Date, String> weekendMap = new TreeMap<Date, String>();
		GregorianCalendar day = new GregorianCalendar(year, 0, 1);
		int daysInYear = day.isLeapYear(year)? 366: 365;
		for (int i = 0; i < daysInYear; i++) {
			day.set(Calendar.DAY_OF_YEAR, (i + 1));
			if (day.get(Calendar.DAY_OF_WEEK) == Calendar.SATURDAY) {
				weekendMap.put(day.getTime(), "Saturday");
			}
			if (day.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY) {
				weekendMap.put(day.getTime(), "Sunday");
			}
		}
		return weekendMap;
	}

	/**
	 * Returns  the day of week of the given date.
	 * @param date The day to retrieve the day of week from.
	 * @return the day of week of the given date.
	 */
	public static String getDayOfWeek(GregorianCalendar date) {
		int dayOfWeek = date.get(Calendar.DAY_OF_WEEK);
		String result;
		switch (dayOfWeek) {
			case Calendar.MONDAY:
				result = "Monday";
				break;
			case Calendar.TUESDAY:
				result = "Tuesday";
				break;
			case Calendar.WEDNESDAY:
				result = "Wednesday";
				break;
			case Calendar.THURSDAY:
				result = "Thursday";
				break;
			case Calendar.FRIDAY:
				result = "Friday";
				break;
			case Calendar.SATURDAY:
				result = "Saturday";
				break;
			case Calendar.SUNDAY:
				result = "Sunday";
				break;
			default:
				result = "Unknown Day";
		}
		return result;
	}

	/**
	 * Returns a sorted array of
	 * @param dates The list of dates. All objects must be of type <code>java.util.Date</code>.
	 * @return the sorted dates
	 */
	public static Date[] sortByDayOfWeek(List<GregorianCalendar> dates) {
		TreeSet<Date> resultSet = new TreeSet<Date>(new Comparator<Date>() {
			public int compare(Date o1, Date o2) {
				GregorianCalendar date1 = new GregorianCalendar();
				date1.setTime(o1);
				GregorianCalendar date2 = new GregorianCalendar();
				date1.setTime(o2);
				int result = 0;
				if (date1.get(Calendar.DAY_OF_WEEK) < date2.get(Calendar.DAY_OF_WEEK)) {
					result = -1;
				} else if (date1.get(Calendar.DAY_OF_WEEK) > date2.get(Calendar.DAY_OF_WEEK)) {
					result = 1;
				}
				return result;
			}
		});
		int length = dates.size();
		for (int i = 0; i < length; i++) {
			resultSet.add((dates.get(i)).getTime());
		}
		Date[] result = new Date[resultSet.size()];
		resultSet.toArray(result);
		return result;
	}

	public static Set<Date> sortByDayOfYear(Date[] dates) {
		TreeSet<Date> resultSet = new TreeSet<Date>(new Comparator<Date>() {
			public int compare(Date o1, Date o2) {
				GregorianCalendar date1 = new GregorianCalendar();
				date1.setTime(o1);
				GregorianCalendar date2 = new GregorianCalendar();
				date2.setTime(o2);
				int result = 0;
				if (date1.get(Calendar.DAY_OF_YEAR) < date2.get(Calendar.DAY_OF_YEAR)) {
					result = -1;
				} else if (date1.get(Calendar.DAY_OF_YEAR) > date2.get(Calendar.DAY_OF_YEAR)) {
					result = 1;
				}
				return result;
			}
		});
        resultSet.addAll(Arrays.asList(dates));
		return resultSet;
	}
}
