/*
 * Copyright: Scheer E2E AG
 */
package ch.e2e.bridge.server.office;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.WorkbookFactory;

import java.io.*;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

@SuppressWarnings("UnusedDeclaration")
public class SpreadsheetAdapter {
    public static final String DEFAULT_ENCODING = System.getProperty("file.encoding", "utf-8");
    public static final String LINE_SEPARATOR = System.getProperty("line.separator", "\n");

    /**
     * Creates a {@link Workbook} object from the Excel document referenced by {@code filename}.
     * Empty cells/rows are omitted.
     *
     * @param filename the filename of the Excel document
     * @return a {@link Workbook} object
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static Workbook createWorkbook(String filename) throws IOException, InvalidFormatException {
        return createWorkbook(new File(filename));
    }

    /**
     * Creates a {@link Workbook} object from the Excel document referenced by {@code filename}.
     *
     * @param filename       the filename of the Excel document
     * @param omitEmptyCells if true empty cells/rows are omitted.
     * @return a {@link Workbook} object
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static Workbook createWorkbook(String filename, boolean omitEmptyCells) throws IOException, InvalidFormatException {
        return createWorkbook(new File(filename), omitEmptyCells);
    }

    /**
     * Creates a {@link Workbook} object from an Excel document.
     * Empty cells/rows are omitted.
     *
     * @param input the content of the Excel document
     * @return a {@link Workbook} object
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static Workbook createWorkbook(byte[] input) throws IOException, InvalidFormatException {
        return createWorkbook(new ByteArrayInputStream(input));
    }

    /**
     * Creates a {@link Workbook} object from an Excel document.
     *
     * @param input          the content of the Excel document
     * @param omitEmptyCells if true empty cells/rows are omitted.
     * @return a {@link Workbook} object
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static Workbook createWorkbook(byte[] input, boolean omitEmptyCells) throws IOException, InvalidFormatException {
        return createWorkbook(new ByteArrayInputStream(input), omitEmptyCells);
    }

    private static Workbook createWorkbook(InputStream input) throws IOException, InvalidFormatException {
        return new Workbook(WorkbookFactory.create(input));
    }

    private static Workbook createWorkbook(InputStream input, boolean omitEmptyCells) throws IOException, InvalidFormatException {
        return new Workbook(WorkbookFactory.create(input), omitEmptyCells);
    }

    private static Workbook createWorkbook(File input) throws IOException, InvalidFormatException {
        return new Workbook(WorkbookFactory.create(input));
    }

    private static Workbook createWorkbook(File input, boolean omitEmptyCells) throws IOException, InvalidFormatException {
        return new Workbook(WorkbookFactory.create(input), omitEmptyCells);
    }

    /**
     * Converts an Excel document to CSV data. This method makes the following assumptions:
     * <list>
     * <li>1. If the Excel workbook contains more that one worksheet, then only the
     * first one will be converted.</li>
     * <li>2. The data matrix contained in the CSV file will be square. This means that
     * the number of fields in each record of the CSV file will match the number
     * of cells in the longest row found in the Excel workbook. Any short records
     * will be 'padded' with empty fields - an empty field is represented in the
     * the CSV file in this way - ,,.</li>
     * <li>3. Empty fields will represent missing cells.</li>
     * <li>4. A record consisting of empty fields will be used to represent an empty row
     * in the Excel workbook.</li>
     * </list>
     * Therefore, if the worksheet looked like this;
     * <p/>
     * <pre>
     *  ___________________________________________
     *     |       |       |       |       |       |
     *     |   A   |   B   |   C   |   D   |   E   |
     *  ___|_______|_______|_______|_______|_______|
     *     |       |       |       |       |       |
     *   1 |   1   |   2   |   3   |   4   |   5   |
     *  ___|_______|_______|_______|_______|_______|
     *     |       |       |       |       |       |
     *   2 |       |       |       |       |       |
     *  ___|_______|_______|_______|_______|_______|
     *     |       |       |       |       |       |
     *   3 |       |   A   |       |   B   |       |
     *  ___|_______|_______|_______|_______|_______|
     *     |       |       |       |       |       |
     *   4 |       |       |       |       |   Z   |
     *  ___|_______|_______|_______|_______|_______|
     *     |       |       |       |       |       |
     *   5 | 1,400 |       |  250  |       |       |
     *  ___|_______|_______|_______|_______|_______|
     *
     * </pre>
     * <p/>
     * Then, the resulting CSV file will contain the following lines (records);
     * <pre>
     * 1,2,3,4,5
     * ,,,,
     * ,A,,B,
     * ,,,,Z
     * "1,400",,250,,
     * </pre><p>
     * The comma is used to separate each of the fields that, together,
     * constitute a single record or line within the CSV file.
     * </p><p>
     * If a field contains the separator then it will be escaped.
     * </p><p>
     * If a field contains an end of line (EOL) character then it too will be
     * escaped.
     * </p><p>
     * If the field contains double quotes then that character will be escaped. An
     * enclosing set of speech marks will also surround the entire field. Thus, if
     * the following line of text appeared in a cell - "Hello" he said - it would
     * look like this when converted into a field within a CSV file - """Hello"" he
     * said".
     * </p>
     *
     * @param filename the filename of the Excel document
     * @return UTF-8 encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename) throws IOException, InvalidFormatException {
        return toCSV(filename, 0, CSVConverter.DEFAULT_SEPARATOR, DEFAULT_ENCODING);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies. But the separator
     * is given by {@code separator}.
     *
     * @param filename  the filename of the Excel document
     * @param separator the separator used to separate the fields
     * @return UTF-8 encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename, String separator) throws IOException, InvalidFormatException {
        return toCSV(filename, 0, separator, DEFAULT_ENCODING);
    }


    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies. But
     * <list>
     * <li>- the separator is given by {@code separator}</li>
     * <li>- the encoding is given by {@code encoding}</li>
     * </list>
     *
     * @param filename  the filename of the Excel document
     * @param separator the separator used to separate the fields
     * @param encoding  the encoding
     * @return {@code encoding} encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename, String separator, String encoding) throws IOException, InvalidFormatException {
        return toCSV(filename, 0, separator, encoding);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies. But
     * <list>
     * <li>- the worksheet specified by {@code sheetNumber} is converted to CSV data</li>
     * </list>
     *
     * @param filename    the filename of the Excel document
     * @param sheetNumber the number of the worksheet to be converted
     * @return UTF-8 encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename, int sheetNumber) throws IOException, InvalidFormatException {
        return new CSVConverter(new File(filename)).convertToCVS(sheetNumber).getBytes();
    }


    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies. But
     * <list>
     * <li>- the worksheet specified by {@code sheetNumber} is converted to CSV data</li>
     * <li>- the separator is given by {@code separator}</li>
     * </list>
     *
     * @param filename    the filename of the Excel document
     * @param sheetNumber the number of the worksheet to be converted
     * @param separator   the separator used to separate the fields
     * @return UTF-8 encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename, int sheetNumber, String separator) throws IOException, InvalidFormatException {
        return new CSVConverter(new File(filename), separator).convertToCVS(sheetNumber).getBytes();
    }


    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies. But
     * <list>
     * <li>- the worksheet specified by {@code sheetNumber} is converted to CSV data</li>
     * <li>- the separator is given by {@code separator}</li>
     * <li>- the encoding is given by {@code encoding}</li>
     * </list>
     *
     * @param filename    the filename of the Excel document
     * @param sheetNumber the number of the worksheet to be converted
     * @param separator   the separator used to separate the fields
     * @param encoding    the encoding
     * @return {@code encoding} encoded CSV data
     * @throws IOException
     * @throws InvalidFormatException
     */
    public static byte[] toCSV(String filename, int sheetNumber, String separator, String encoding) throws IOException, InvalidFormatException {
        return new CSVConverter(new File(filename), separator).convertToCVS(sheetNumber).getBytes(encoding);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies.
     *
     * @param workbook the {@link Workbook} object referencing the Excel document.
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook) throws IOException {
        return toCSV(workbook, CSVConverter.DEFAULT_SEPARATOR, DEFAULT_ENCODING);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, String)} applies.
     *
     * @param workbook  the {@link Workbook} object referencing the Excel document.
     * @param separator the separator used to separate the fields
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook, String separator) throws IOException {
        return toCSV(workbook, separator, DEFAULT_ENCODING);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(Sheet, String, String)} applies.
     *
     * @param workbook  the {@link Workbook} object referencing the Excel document.
     * @param separator the separator used to separate the fields
     * @param encoding  the encoding
     * @return {@code encoding} encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook, String separator, String encoding) throws IOException {
        return toCSV(workbook, 0, separator, encoding);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, int)} applies.
     *
     * @param workbook    the {@link Workbook} object referencing the Excel document.
     * @param sheetNumber the number of the worksheet to be converted
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook, int sheetNumber) throws IOException {
        return toCSV(workbook.getSheet(sheetNumber), CSVConverter.DEFAULT_SEPARATOR);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, int, String)} applies.
     *
     * @param workbook    the {@link Workbook} object referencing the Excel document.
     * @param sheetNumber the number of the worksheet to be converted
     * @param separator   the separator used to separate the fields
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook, int sheetNumber, String separator) throws IOException {
        return toCSV(workbook.getSheet(sheetNumber), separator);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, int, String, String)} applies.
     *
     * @param workbook    the {@link Workbook} object referencing the Excel document.
     * @param sheetNumber the number of the worksheet to be converted
     * @param separator   the separator used to separate the fields
     * @param encoding    the encoding
     * @return {@code encoding} encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Workbook workbook, int sheetNumber, String separator, String encoding) throws IOException {
        return toCSV(workbook.getSheet(sheetNumber), separator, encoding);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String)} applies.
     *
     * @param sheet the {@link Sheet} object referencing the Excel worksheet
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Sheet sheet) throws IOException {
        return toCSV(sheet, CSVConverter.DEFAULT_SEPARATOR, DEFAULT_ENCODING);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, String)} applies.
     *
     * @param sheet     the {@link Sheet} object referencing the Excel worksheet
     * @param separator the separator used to separate the fields
     * @return UTF-8 encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Sheet sheet, String separator) throws IOException {
        return toCSV(sheet, separator, DEFAULT_ENCODING);
    }

    /**
     * The description of {@link SpreadsheetAdapter#toCSV(String, String, String)} applies.
     *
     * @param sheet     the {@link Sheet} object referencing the Excel worksheet
     * @param separator the separator used to separate the fields
     * @param encoding  the encoding
     * @return {@code encoding} encoded CSV data
     * @throws IOException
     */
    public static byte[] toCSV(Sheet sheet, String separator, String encoding) throws IOException {
        if (separator == null) separator = CSVConverter.DEFAULT_SEPARATOR;
        if (encoding == null) encoding = DEFAULT_ENCODING;
        ByteArrayOutputStream os = new ByteArrayOutputStream(1024);
        final Iterator<Row> iterator = sheet.getRows().iterator();
        if (iterator.hasNext()) {
            String line = rowToCSV(iterator.next(), sheet.getNumberOfColumns(), separator);
            os.write(line.getBytes(encoding));
            while (iterator.hasNext()) {
                os.write(LINE_SEPARATOR.getBytes(encoding));
                line = rowToCSV(iterator.next(), sheet.getNumberOfColumns(), separator);
                os.write(line.getBytes(encoding));
            }
        }
        return os.toByteArray();
    }

    private static String rowToCSV(Row row, int numberOfColumns, String separator) {
        List<Cell> cells = (row == null ? Collections.<Cell>emptyList() : row.getCells());
        return join(new CellIterator(cells, numberOfColumns), separator);
    }

    private static String join(CellIterator iterator, String separator) {
        String result = "";
        if (iterator.hasNext()) {
            Cell next = iterator.next();
            String value;
            StringBuilder builder = (next == null || (value = next.getValue()) == null
                                     ? new StringBuilder()
                                     : new StringBuilder(CSVConverter.escapeEmbeddedCharacters(value, separator, CSVConverter.Escaping.EXCEL_STYLE)));
            while (iterator.hasNext()) {
                builder.append(separator);
                next = iterator.next();
                if (next != null && (value = next.getValue()) != null) {
                    builder.append(CSVConverter.escapeEmbeddedCharacters(value, separator, CSVConverter.Escaping.EXCEL_STYLE));
                }
            }
            result = builder.toString();
        }
        return result;
    }

    private SpreadsheetAdapter() {
    }

    private static class CellIterator implements Iterator<Cell> {
        private final Iterator<Cell> cells;
        private final int size;
        private int cursor = 0;

        private CellIterator(Iterable<Cell> cells, int size) {
            this.cells = cells.iterator();
            this.size = size;
        }

        @Override
        public boolean hasNext() {
            return cursor < size;
        }

        @Override
        public Cell next() {
            cursor++;
            if (cells.hasNext()) {
                return cells.next();
            } else {
                return null;
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
