import { BrowserModule } from '@angular/platform-browser';
import { APP_INITIALIZER, NgModule } from '@angular/core';
import { BrowserAnimationsModule } from '@angular/platform-browser/animations';

import { AppRoutingModule } from './app.routing.module';
import { AppComponent } from './app.component';
import { ProcessListComponent } from "./components/process-list/process-list.component";
import { HttpClientModule, HttpClient } from '@angular/common/http';
import { TranslateLoader, TranslateModule } from '@ngx-translate/core';
import { TranslateHttpLoader } from '@ngx-translate/http-loader';
import { MaterialModule } from './material.module';
import { FormsModule } from './forms/forms.module';
import { UserComponent } from "./components/user/user.component";
import { FontAwesomeModule } from '@fortawesome/angular-fontawesome';
import {AppCoreModule, FormService as CoreFormService, SecurityService as CoreSecurityService} from "@pas/app-core";
import { FormElementsModule } from "@pas/form-elements";
import {XumlService} from "@pas/xuml-communication";
import {
    HelperService,
    PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL,
    KeyCloakHelperService,
    KEYCLOAK_SESSION_STORAGE_TOKEN
} from "@pas/platform-communication";
import {FormService} from "./services/form.service";
import {SecurityService} from "./services/security.service";
import {LibrariesModule} from "./libraries.module";
import {AppConfig} from "./app-config";
import { AuthenticationService, PLATFORM_BASE_URL } from '@pas/app-api';
import Axios from 'axios';
import * as JSON5 from 'json5';

// AoT requires an exported function for factories
export function HttpLoaderFactory(http: HttpClient) {
    return new TranslateHttpLoader(http, './assets/i18n/', '.json');
}

@NgModule({
    declarations: [
        AppComponent,
        ProcessListComponent,
        UserComponent
    ],
    imports: [
        AppCoreModule,
        MaterialModule,
        FontAwesomeModule,
        FormElementsModule,
        BrowserModule,
        BrowserAnimationsModule,
        AppRoutingModule,
        HttpClientModule,
        TranslateModule.forRoot({
            loader: {
                provide: TranslateLoader,
                useFactory: HttpLoaderFactory,
                deps: [HttpClient]
            },
            defaultLanguage: 'en'
        }),
        LibrariesModule,
        FormsModule
    ],
    providers: [
        AppComponent,
        {provide: CoreFormService, useClass: FormService},
        {provide: CoreSecurityService, useClass: SecurityService},
        {
            provide: PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL,
            useFactory: (config: AppConfig) => {
                return (config.platformBaseUrl != null) ? config.platformBaseUrl : HelperService.getBaseUrl();
            },
            deps: [AppConfig]
        },
        {
            provide: XumlService,
            useFactory: (config: AppConfig, platformBaseUrl) => {
                let serviceBaseUrl = undefined;
                if(config.xuml.url){
                    serviceBaseUrl = config.xuml.url;
                } else if(config.xuml.platformRelativeUrl){
                    serviceBaseUrl = platformBaseUrl + config.xuml.platformRelativeUrl
                }
                return new XumlService(serviceBaseUrl);
            },
            deps: [AppConfig, PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL]

        },
        {
            provide: APP_INITIALIZER,
            useFactory: (appConfig: AppConfig) => {
                return async () => {
                    const configUrl = 'assets/config/xuml.json5';

                    try {
                        const config = (await Axios.get(configUrl)).data;

                        if(typeof config === 'string'){
                            try {
                                const parsedConfig = JSON5.parse(config);
                                if (parsedConfig.pasAppOptions?.keyCloakOptions != null) {
                                    if (parsedConfig.pasAppOptions.keyCloakOptions.resource != null) {
                                        appConfig.keyCloakOptions.keycloakClientId = parsedConfig.pasAppOptions.keyCloakOptions.resource;
                                    }
                                    if (parsedConfig.pasAppOptions.keyCloakOptions.realm != null) {
                                        appConfig.keyCloakOptions.realm = parsedConfig.pasAppOptions.keyCloakOptions.realm;
                                    }
                                    if (parsedConfig.pasAppOptions.keyCloakOptions['auth-server-url'] != null) {
                                        appConfig.keyCloakOptions.keycloakURL = parsedConfig.pasAppOptions.keyCloakOptions['auth-server-url'];
                                    }
                                }
                            } catch (err) {
                                console.error('cannot parse or merge config: ', config);
                            }
                        }
                    } catch (err) {
                        console.log(`Could not load config from '${configUrl}'`, err);
                    }
                };
            },
            multi: true,
            deps: [AppConfig]
        },
        {
            provide: APP_INITIALIZER,
            useFactory: initializeKeycloak,
            multi: true,
            deps: [AppConfig, KeyCloakHelperService, XumlService]
        },
        {
            provide: AuthenticationService,
            useValue: {
                token: {
                    get value(): string | null { return  window.sessionStorage.getItem(KEYCLOAK_SESSION_STORAGE_TOKEN) }
                }
            }
        },
        {
            provide: PLATFORM_BASE_URL,
            useExisting: PAS_PLATFORM_COMMUNICATION_SERVICE_BASE_URL
        },
    ],
    bootstrap: [AppComponent]
})
export class AppModule { }

function initializeKeycloak(config: AppConfig, keycloakHelper: KeyCloakHelperService, xumlService: XumlService) {
    return async () => {
        try {
            await keycloakHelper.initKeycloak({
                ...{ onLoad: 'login-required' },
                ...config.keyCloakOptions
            });
            if(!config.xuml.isAuthenticator) {
                xumlService.addAxiosInterceptor(keycloakHelper.getAxiosInterceptor());
            }
        } catch (error) {
            console.error('Error initializing Keycloak', error);
        }
    };
}
