import { Component, OnInit} from '@angular/core';
import { ActivatedRoute, Router, NavigationExtras } from '@angular/router';
import {ProgressService, NotificationService, SecurityService, ServerSideFormEvent} from '@pas/app-core';
import {ProcessService} from '@pas/xuml-communication';
import {DataTableComponentAction, DataTableComponentColumnModel} from "@pas/form-elements";
import {TranslateService} from "@ngx-translate/core";
import {FormControl} from "@angular/forms";
import {columnModel} from "./instanceList";

@Component({
  templateUrl: './process.component.html'
})
export class ProcessComponent implements OnInit {
  public processName = '';
  public actions: DataTableComponentAction[] = [];
  public startEvents: string[] = [];
  private roles: any = {};

  public rolesExternalFilter: string[] = [];

  rolesProperties = {
    label: '',
    readonly: false,
    mandatory: false,
    cssClass: '',
    orientation: 0,
    optionsConfiguration: {
      options: [
        {
          badge: null,
          label: 'All',
          value: '__all__'
        }
      ],
      defaultKey: 0
    }
  };

  rolesFormControl = new FormControl('__all__');

  public columnModel: DataTableComponentColumnModel[] = columnModel;

  constructor(
      private route: ActivatedRoute,
      private router: Router,
      private process: ProcessService,
      private notification: NotificationService,
      private progress: ProgressService,
      private translate: TranslateService,
      private security: SecurityService
  ) {
    this.router.onSameUrlNavigation = 'reload';

    this.columnModel.forEach(column => {
        switch (column.name) {
            case '_meta_currentTasks':
                column.getValue = (item) => item._meta.currentTasks.join(', ');
                break;
            case '_meta_creation':
                column.getValue = (item) => new Date(item._meta.creation);
                break;
            case '_meta_lastUpdate':
                column.getValue = (item) => new Date(item._meta.creation);
                break;
        }
    });
  }

  ngOnInit(): void {
    this.route.data.subscribe(data => {
      this.processName = data.processName;
      this.startEvents = data.startEvents;
      this._initActions(data.startEvents);
      this.roles = data.roles.value;
    });
  }

  private _initActions(startEvents: string[]): void {
      const actions: DataTableComponentAction[] = [];
      actions.push({
        key: 'refresh',
        icon: 'refresh',
        actions: []
      });

      let startActions: DataTableComponentAction[] = [];
      startEvents.forEach(key => {
        const resourceId = "process/" + this.processName + "/" + key;
        if (this.security.isAuthorized(resourceId, "write")) {
          startActions.push({key: key, icon: 'arrow_right_alt', actions: []});
        }
      });
      if (startActions.length > 0) {
        actions.push({
          key: 'startEvents',
          icon: 'add',
          actions: startActions
        });
      }
      this.actions = actions;
    }

  public onAction(event): void {
    this._onStartEvent(event)
  }

  public onRowClick(event): void {
    this._navigateTo(event.id);
  }

  private _onStartEvent(event): void {
    for (const s of this.startEvents) {
      if (s === event.key) {
        this._navigateTo(event.key, {
            replaceUrl: false,
            skipLocationChange: true
        });
        return;
      }
    }
  }

  private _navigateTo(to: string, extras?: NavigationExtras): void {
    this.router.navigate([this.processName, to], extras);
  }

  async reloadInstances(event?: ServerSideFormEvent) {
    try {

      let instanceOptions = {...event?.data, roles: event?.data.externalFilterOptions};

      const instanceList = await this.process.getInstanceListExtended(this.processName, instanceOptions);

      if (instanceList == null) {
        this.notification.sendError('Process not found');
        await this.router.navigate([]);
        return;
      }

      event.resolve({
        items: instanceList.instances,
        filteredCount: instanceList.filteredCount,
        totalCount: instanceList.totalCount
      });

      let roles = [];
      let totalInstanceCount = 0;
      Object.keys(this.roles).forEach(name => {
        if (this.security.hasRole(name)) {
          instanceList.roleCounters.forEach(role => {
            if (role.id === '_all_') {
              totalInstanceCount = role.listCount;
            }
            else if (role.id === name) {
              roles.push({
                badge: role.listCount <= 99 ? role.listCount : '99+',
                label: name,
                value: name
              });
            }
          });
        }
      });

      this.rolesProperties.optionsConfiguration.options = [];
      this.rolesProperties.optionsConfiguration.options.push({
        badge: totalInstanceCount <= 99 ? totalInstanceCount : '99+',
        label: 'All',
        value: '__all__'
      }, ...roles);
    }
    catch (error) {
      this.notification.sendNotification({
        type: 'error',
        message: error.message,
        data: {
          error: error
        }
      });
      await this.router.navigate([]);
    }

  }

  updateRolesExternalFilter(): void {
    if (this.rolesFormControl.value != '__all__') {
      this.rolesExternalFilter = [this.rolesFormControl.value];
    } else {
      this.rolesExternalFilter = [];
    }
  }

}
