import {Injectable} from '@angular/core';
import {ActivatedRoute, ActivatedRouteSnapshot, CanActivate, Router, RouterStateSnapshot, UrlTree} from '@angular/router';
import {NotificationService, SecurityService} from "@pas/app-core";
import {InstanceMetadata} from "@pas/xuml-communication";

@Injectable({
    providedIn: 'root'
})
export class TaskGuard implements CanActivate {

    constructor(
        private router: Router,
        private route:  ActivatedRoute,
        private notification: NotificationService,
        private security: SecurityService
    ) {
        this.router.onSameUrlNavigation = 'reload';
    }

    async canActivate(
        routeSnapshot: ActivatedRouteSnapshot,
        state: RouterStateSnapshot): Promise<boolean | UrlTree> {
        const processName = routeSnapshot.data.processName;
        const instanceId = routeSnapshot.params.instanceId;
        const taskName = routeSnapshot.params.taskName;
        const instance: InstanceMetadata = routeSnapshot.data.instance.value;

        try {
            const authorizedTasks = instance._meta.currentTasks.filter(name => {
                const resourceId = "process/" + routeSnapshot.data.processName + "/" + name;
                return (this.security.isAuthorized(resourceId, "read") || this.security.isAuthorized(resourceId, "write"));
            });

            if (taskName == null) {
                if (authorizedTasks.length > 1) {
                    return this.router.navigate([processName, instanceId, 'select-task'], {
                        relativeTo: this.route,
                        replaceUrl: true
                    });
                } else if (authorizedTasks[0] != null) {
                    return this.router.navigate([processName, instanceId, instance._meta.currentTasks[0]], {
                        relativeTo: this.route,
                        replaceUrl: true
                    });
                }
            } else if (authorizedTasks.indexOf(taskName) < 0) {
                return this.router.navigate([processName, instanceId], {
                    relativeTo: this.route,
                    replaceUrl: true
                });
            }

        } catch (error) {
            this.notification.sendError(error.message);
            return this.router.navigate([processName], {
                relativeTo: this.route,
                replaceUrl: true
            });
        }

        return true;
    }

}