import {Component, HostBinding, Inject} from '@angular/core';
import {MatSnackBar} from '@angular/material/snack-bar';
import {ActivatedRoute, NavigationEnd, Router} from '@angular/router';
import {NotificationService, ProgressService, UserMenuOptionSettings, SnackbarComponent} from '@pas/app-core';
import {AuthService, CdnService, CurrentUserService, IUserModel, StorageService} from '@pas/platform-communication';
import {AppRoutingModule, Navigation} from './app.routing.module';
import {DOCUMENT} from "@angular/common";
import {filter} from "rxjs/operators";
import {TranslateService} from '@ngx-translate/core';

@Component({
  selector: 'app-root',
  templateUrl: './app.component.html'
})
export class AppComponent {

  @HostBinding('style') style;
  @HostBinding('class') class;

  user: IUserModel;
  userSettings: UserMenuOptionSettings;

  navigation: Navigation[];

  caption: string | null = null;

  inProgress = false;


  constructor(
      private router: Router,
      private route: ActivatedRoute,
      public progress: ProgressService,
      public notification: NotificationService,
      public snackbar: MatSnackBar,
      private routing: AppRoutingModule,
      private currentUserService: CurrentUserService,
      private authService: AuthService,
      private cdn: CdnService,
      private storage: StorageService,
      private translateService: TranslateService,
      @Inject(DOCUMENT) private document: Document
  ) {
    this.initStyles();
    this.initSnackbar();
    this.initSpinner();
    this.initUser();
    this.initNavigation();
  }

  private async initUser() {
    this.userSettings = {
        showManual: false,
        showAcademy: false,
        showLogout: true,
        showI18n: true,
        showPreferences: true
    };
    this.router.events
        .pipe(filter(event => event instanceof NavigationEnd))
        .subscribe(() => {
          this.user = this.route.firstChild.snapshot.data.user.value;
        });
  }

  private initSnackbar() {
    this.notification.notification.subscribe((notification) => {
      if(notification){
        let { message, type, data } = notification;

        let errorMessage = message;
        if (data && data.error && data.error.isAxiosError && data.error.isAxiosError === true) {

          let httpStatusCode = data.error.response.status;

          if (httpStatusCode === 404) {
            if (data.type && data.type === 'ProcessInstance') {
              errorMessage = this.translateService.instant('errors.404');
              type = 'success';
            } else {
              errorMessage = this.translateService.instant('errors.4??');
              type = 'error';
            }
          } else if (httpStatusCode === 401) {
            errorMessage = this.translateService.instant('errors.401');
          } else if (httpStatusCode >= 400 && httpStatusCode < 500) {
            errorMessage = this.translateService.instant('errors.4??');
          } else if (httpStatusCode >= 500 && httpStatusCode <= 511) {
            errorMessage = this.translateService.instant('errors.5??');
          }
        }

        this.snackbar.openFromComponent(SnackbarComponent, {
          duration: type === 'error' ? 5000 : 2000,
          data: { message: errorMessage, type}
        });
      }
    });
  }

  private initSpinner() {
    this.progress.inProgress.subscribe( (inProgress) => {
      this.inProgress = inProgress;
      this.style = this.inProgress ? 'pointer-events: none;' : undefined;
      this.class = {
        'in-progress': this.inProgress
      };
    });
  }

  private initNavigation() {
    this.navigation = this.routing.navigation;
    this.router.events.subscribe((val) => {
      if ( val instanceof NavigationEnd ) {
        const { url } = val;
        const split = url.split('/');
        const current = this.navigation.find((item) => item.path === split[1])
            || this.navigation.find((item, index) => index === 0);
        this.caption = current.label;
      }
    });
  }

  private async initStyles() {
    const styles = await import('./css.json');

    for(const css of styles.css){
      switch (css.type) {
        case 'cdn':
          this._loadStyle(await this.cdn.getFileUrl(css.namespace, css.path));
          break;
        case 'storage':
          this._loadStyle(await this.storage.getFileUrl(css.namespace, css.path));
          break;
      }
    }
  }

  private _loadStyle(linkPath: string) {
    const head = this.document.getElementsByTagName('head')[0];
    const links = head.getElementsByTagName('link');

    for(let i = 0; i < links.length; i++){
      const link = links.item(i);
      if (link.href == linkPath){
        return;
      }
    }

    const style = this.document.createElement('link');
    style.rel = 'stylesheet';
    style.href = `${linkPath}`;
    head.appendChild(style);
  }

  public logout(): void {
    this.authService.logout();
  }

}
