/*
 * Copyright: Scheer PAS GmbH
 */
package com.scheer.pas.simple;

import ch.e2e.bridge.server.BridgeException;
import ch.e2e.bridge.server.Logger;
import ch.e2e.bridge.server.XumlLibraryClass;

import static ch.e2e.bridge.server.Logger.*;

/*
 * This is a simple Java implementation example that can be compiled to an xUML library for the Designer.
 *
 * This implementation demonstrates several important concepts:
 *   - the class must be annotated with '@XumlLibraryClass' to be recognized by the xUML Compiler
 *   - methods must be 'public static' to be recognized by the xUML Compiler
 */

/**
 * A utility class for demonstrating simple operations such as returning a message
 * or throwing an exception. Includes methods for throwing a {@code BridgeException}
 * with or without a provided cause.
 */
@SuppressWarnings("unused")
@XumlLibraryClass(name = "HelloWorld")
public class HelloWorldExample {

    private static final Logger LOGGER = new Logger();

    private HelloWorldExample() {
    }

    /**
     * Returns the provided message if it is not null or empty. Otherwise, returns a default message.
     * Logs the resolved message for debugging purposes.
     *
     * @param message The message to be returned. If null or empty, a default message is returned.
     * @return The resolved message, which is either the provided message or a default message if the input is null
     * or empty.
     */
    public static String helloWorld(String message) {
        String msg = message == null || message.isEmpty() ? "Hi from the Scheer PAS team!" : message;
        LOGGER.log(LL_DEBUG, "Internal", "com.scheer.pas.simple", "999", "helloWorld() - answer is '" + msg + "'.");
        return msg;
    }

    /**
     * Generates a birthday greeting message for the provided person.
     *
     * @param person The Person object containing the name and age of the individual for whom the greeting will be created.
     * @return A string containing the personalized birthday greeting message.
     */
    public static String birthdayGreetings(Person person) {
        return String.format("Happy %s birthday, dear %s!", person.getAgeWithSuffix(), person.getFirstName());
    }

    /**
     * Throws a {@code BridgeException} with the specified message, error code, and error domain.
     *
     * @param message The description of the error to be included in the exception.
     * @param code The code representing the specific error.
     * @param domain The domain or context in which the error occurred.
     */
     public static void throwException(String message, String code, String domain) {
        throw new BridgeException(message, code, domain);
    }

    /**
     * Throws a {@code BridgeException} with a specified cause, error code, and error domain.
     *
     * @param message The description of the error to be included in the exception's cause.
     * @param code The code representing the specific error.
     * @param domain The domain or context in which the error occurred.
     */
    public static void throwExceptionWithCause(String message, String code, String domain) {
        throw new BridgeException(new IllegalStateException(message), code, domain);
    }

}
