/*
 * Copyright: Scheer PAS GmbH
 */

package com.scheer.pas.service;

import ch.e2e.bridge.server.BridgeJavaService;
import ch.e2e.bridge.server.BridgeJavaServiceStartStopInterface;
import ch.e2e.bridge.server.XumlLibraryClass;
import com.sun.net.httpserver.HttpServer;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/*
 * This is an example for a PAS xlib Java service.
 *
 * This implementation demonstrates several important concepts of such a class:
 *   - it must be annotated with '@XumlLibraryClass' to be recognized by the xUML Compiler
 *   - it must implement 'BridgeJavaService'
 *   - it can implement 'BridgeJavaServiceStartStopInterface'
 */

/**
 * The SimpleHttpServer class represents an HTTP server implementation for serving requests.
 * It uses callbacks defined in the SimpleHttpServerCallback interface to define custom
 * behavior for request handling and response creation.
 * <p>
 * The class initializes an HTTP server, starts it to accept incoming requests, and allows
 * gracefully stopping the server when required.
 * <p>
 * This implementation interacts with a callback instance of SimpleHttpServerCallback to:<br>
 * - Retrieve the port number for the server.<br>
 * - Create response bodies for incoming requests.
 */
@SuppressWarnings("unused")
@XumlLibraryClass(name = "HttpServer")
public class SimpleHttpServer
        implements BridgeJavaService<SimpleHttpServerCallback>, BridgeJavaServiceStartStopInterface {

    public static final int SERVER_STOP_DELAY_SECONDS = 5;

    private SimpleHttpServerCallback callback;

    private HttpServer httpServer;
    private ExecutorService executor;

    /**
     * Initialize the service; should not access other components or may cause race conditions.
     * DO NOT call callback methods here.
     *
     * @param callback parameter with one or more member functions having an activity diagram
     */
    @Override
    public void initialize(SimpleHttpServerCallback callback) {
        this.callback = callback;
        try {
            httpServer = HttpServer.create();
            httpServer.createContext("/", exchange -> {
                byte[] responseBody = callback.createResponseBody(exchange.getRequestMethod(),
                                                                  exchange.getRequestURI().toASCIIString(),
                                                                  exchange.getRequestBody());
                exchange.sendResponseHeaders(200, responseBody.length);
                try (OutputStream os = exchange.getResponseBody()) {
                    os.write(responseBody);
                }
            });
            executor = Executors.newSingleThreadExecutor();
            httpServer.setExecutor(executor);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Stop accepting new requests, wait for all active processing to finish.
     * DO NOT call callback methods here.
     */
    @Override
    public void shutDown() {
        try {
            executor.shutdown();
            //noinspection ResultOfMethodCallIgnored
            executor.awaitTermination(SERVER_STOP_DELAY_SECONDS, java.util.concurrent.TimeUnit.SECONDS);
        } catch (InterruptedException ignored) {
        }
        httpServer.stop(SERVER_STOP_DELAY_SECONDS);
    }

    /**
     * Start serving requests. From this point on all components can be freely accessed.
     */
    @Override
    public void start() {
        try {
            httpServer.bind(new InetSocketAddress(callback.getPort()), 0);
            httpServer.start();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Do some necessary cleanup.
     * DO NOT call callback methods here.
     */
    @Override
    public void stop() {
    }

}
