package com.scheer.pas.large;

import ch.e2e.bridge.server.XumlLibraryClass;

import java.util.NoSuchElementException;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/*
 * This example shows how you can reduce the amount of data to be transferred between the xUML Runtime and the JVM
 * when dealing with large objects.
 *
 * This implementation demonstrates several important concepts:
 *   1. Resource management: Images (which can be large) are stored in memory and explicitly released
 *      when no longer needed
 *   2. Thread safety: Using `ConcurrentMap` ensures thread-safe operations on the image collection
 *   3. Unique identification: UUIDs ensure that each image has a unique reference
 *   4. Error handling: Methods include appropriate exception handling for cases where images don't exist
 *   5. Separation of concerns: The class separates metadata access from full data access, allowing optimization
 *      of data transfer
 *
 * This pattern is particularly useful when working with large objects where you want to:
 *   - Minimize data transfer between systems
 *   - Provide a simple reference mechanism (the UUID string)
 *   - Explicitly control the lifecycle of the resource
 *
 * You may use a Java image processing library to perform the 'internal' image handling / processing.
 */

/**
 * A utility class holding image operations, including creating, retrieving, processing, and releasing
 * images using unique identifiers.
 */
@SuppressWarnings("unused")
@XumlLibraryClass(name = "Images")
public class ImageHandling {

    private static final ConcurrentMap<String, Image> IMAGES_MAP = new ConcurrentHashMap<>();

    private ImageHandling() {
    }

   /**
    * Creates a new image from the given byte array of image data and returns a unique identifier for the image.
    *
    * @param imageData The byte array containing the raw data of the image.
    * @return The unique identifier assigned to the created image.
    * @throws IllegalStateException if an image with the generated identifier already exists in the map.
    */
   public static String newImage(byte[] imageData) {
        String id = UUID.randomUUID().toString();
        if (IMAGES_MAP.put(id, new Image(imageData)) != null) {
            throw new IllegalStateException(String.format("Image with id '%s' already exists.", id));
        }
        return id;
    }

    /**
     * Releases the image associated with the specified unique identifier.
     *
     * @param imageId The unique identifier of the image to be removed.
     */
    public static void releaseImage(String imageId) {
        IMAGES_MAP.remove(imageId);
    }

    /**
     * Releases All currently stored images.
     */
    public static void releaseAllImages() {
        IMAGES_MAP.clear();
    }

    /**
     * Retrieves The size of the image identified by the specified unique identifier.
     *
     * @param imageId The unique identifier of the image whose size is to be retrieved.
     * @return The size of the image in bytes.
     * @throws NoSuchElementException if no image with the provided identifier exists.
     */
    public static int getImageSize(String imageId) {
        return getImage(imageId).getSize();
    }

    /**
     * Retrieves the width of the image identified by the specified unique identifier.
     *
     * @param imageId The unique identifier of the image whose width is to be retrieved.
     * @return The width of the specified image.
     * @throws NoSuchElementException if no image with the provided identifier exists.
     */
    public static int getImageWidth(String imageId) {
        return getImage(imageId).getWidth();
    }

    /**
     * Retrieves the height of the image identified by the specified unique identifier.
     *
     * @param imageId The unique identifier of the image whose height is to be retrieved.
     * @return The height of the specified image.
     * @throws NoSuchElementException if no image with the provided identifier exists.
     */
    public static int getImageHeight(String imageId) {
        return getImage(imageId).getHeight();
    }

    /**
     * Processes the image identified by the specified unique identifier.
     * The method retrieves the image data and performs some processing on it.
     *
     * @param imageId The unique identifier of the image to be processed.
     * @throws NoSuchElementException if no image with the provided identifier exists.
     */
    public static void processImage(String imageId) {
        byte[] data = getImage(imageId).getData();
        // Implement some processing of the image data here
    }

    /**
     * Retrieves the image associated with the specified unique identifier.
     *
     * @param imageId The unique identifier of the image to be retrieved.
     * @return The image associated with the given identifier.
     * @throws NoSuchElementException if no image with the provided identifier exists.
     */
    private static Image getImage(String imageId) {
        Image image = IMAGES_MAP.get(imageId);
        if (image == null) {
            throw new NoSuchElementException(String.format("Image with id '%s' not found.", imageId));
        }
        return image;
    }

}
